/**
 * @license Creative Commons Attribution-NonCommercial-ShareAlike 4.0 International (CC BY-NC-SA 4.0)
 *
 * This sketch is licensed under CC BY-NC-SA 4.0. You are free to:
 * - Share and adapt this work
 * - Use modified versions commercially
 *
 * Under these conditions:
 * - Attribution: Credit Ben McCormick (phobon) and link to this project
 * - NonCommercial: Don't sell the original, unmodified sketch
 * - ShareAlike: Distribute derivatives under the same license
 */

import {
  Fn,
  time,
  vec3,
  screenSize,
  smoothstep,
  cos,
  sin,
  vec2,
  mix,
  abs,
  Loop,
  pow,
  normalize,
  div,
  sub,
} from 'three/tsl'
import { screenAspectUV } from '@/tsl/utils/function/screen_aspect_uv'
import { simplexNoise3d } from '@/tsl/noise/simplex_noise_3d'
import { cosinePalette } from '@/tsl/utils/color/cosine_palette'
import { sdSphere } from '@/tsl/utils/sdf/shapes'

/**
 * Genuary 11 - Impossible day - Try to do something that feels impossible for you to do. Maybe it is impossible. Maybe it’s too ambitious. Maybe it’s something you know nothing about how to accomplish
 */
export const genuary11 = Fn(() => {
  // Aspect-corrected UVs
  const _uv = screenAspectUV(screenSize)

  // Palette
  const a = vec3(0.5, 0.5, 0.5)
  const b = vec3(0.5, 0.5, 0.5)
  const c = vec3(1.0, 0.7, 0.4)
  const d = vec3(0.0, 0.15, 0.2)

  // Color accumulator
  const finalColor = vec3(0.0).toVar()

  // Rotating domain for mask
  const rotationAngle = time.mul(0.75)
  const rotatedDomain = _uv.add(vec2(sin(rotationAngle), cos(rotationAngle)).mul(0.3))
  const mask = smoothstep(0.0, 0.75, sdSphere(rotatedDomain.mul(2.0), 0.01))

  // Loop to create layered, warped spheres with noise
  // @ts-ignore
  Loop({ start: 0, end: 15, type: 'float' }, ({ i }) => {
    // Determine the direction the noise is moving in
    const direction = _uv.sub(c).toVar()
    direction.assign(normalize(direction))

    // Apply the noise based on the normalized direction
    const noise = simplexNoise3d(vec3(direction.mul(10.0).xy, time.mul(0.5).add(i)))
    const factor = mix(direction.mul(noise).mul(0.04).xy, vec2(0.0), mask)

    // Create a shape with a bloomed edge
    const shape = sdSphere(_uv.add(factor), 0.4).toVar()
    shape.assign(abs(shape))
    shape.assign(pow(div(0.0001, shape), 1.1))

    // Palette-based color, modulated by mask and time
    const col = cosinePalette(sub(1.0, mask).mul(sdSphere(_uv)).add(i).add(time.mul(0.25)), a, b, c, d)
    finalColor.addAssign(col.mul(shape))
  })

  return finalColor
})
export default genuary11
