import { screenAspectUV } from '@/tsl/utils/function/screen_aspect_uv'
import { sdLine, sdBox2d, sdDiamond, sdSphere, sdRing, sdHexagon, sdEquilateralTriangle } from '@/tsl/utils/sdf/shapes'
import { Fn, step, clamp, rotate, float, max, abs, fract, smoothstep, vec3, screenSize } from 'three/tsl'

/**
 * Makes a plus symbol by combining vertical and horizontal lines, masked by a box SDF.
 * @param {vec2} _uv - The UV coordinates.
 * @param {number} [maskSize=0.25] - Size of the box mask.
 * @param {number} [width=0.98] - Width of the plus lines.
 * @returns {float} Pattern mask value (0-1).
 */
const plusPattern = Fn(([_uv, maskSize = 0.25, width = 0.98]) => {
  // Add the lines together
  const verticalLine = sdLine(_uv.x).oneMinus().toVar()
  verticalLine.assign(step(width, verticalLine))
  const horizontalLine = sdLine(_uv.y).oneMinus().toVar()
  horizontalLine.assign(step(width, horizontalLine))

  verticalLine.addAssign(horizontalLine)

  // Mask the outside to give us a plus symbol
  const mask = sdBox2d(_uv).toVar()
  mask.assign(step(mask, maskSize))

  return mask.mulAssign(clamp(0, 1, verticalLine))
})

/**
 * Produces a diamond-masked cross by combining rotated lines and masking with a diamond SDF.
 * @param {vec2} _uv - The UV coordinates.
 * @param {number} [width=0.98] - Width of the cross lines.
 * @param {number} [maskSize=0.35] - Size of the diamond mask.
 * @returns {float} Pattern mask value (0-1).
 */
const crossPattern = Fn(([_uv, width = 0.98, maskSize = 0.35]) => {
  const rotatedUv = rotate(_uv, 0.25 * Math.PI)

  // Add the lines together
  const verticalLine = sdLine(rotatedUv.x).oneMinus().toVar()
  verticalLine.assign(step(width, verticalLine))
  const horizontalLine = sdLine(rotatedUv.y).oneMinus().toVar()
  horizontalLine.assign(step(width, horizontalLine))

  verticalLine.addAssign(horizontalLine)

  // Mask the outside to give us a cross symbol
  const mask = sdDiamond(_uv).toVar()
  mask.assign(step(mask, maskSize))

  return mask.mulAssign(clamp(0, 1, verticalLine))
})

/**
 * Generates a single diagonal slash, masked by a diamond shape.
 * @param {vec2} _uv - The UV coordinates.
 * @param {number} [maskSize=0.35] - Size of the diamond mask.
 * @param {number} [width=0.98] - Width of the slash line.
 * @returns {float} Pattern mask value (0-1).
 */
const slashPattern = Fn(([_uv, maskSize = 0.35, width = 0.98]) => {
  // Rotate the uvs by 45 degrees
  const rotatedUv = rotate(_uv, 0.25 * Math.PI)

  const slash = sdLine(rotatedUv.x).oneMinus().toVar()
  slash.assign(step(width, slash))

  // Mask the outside of the slash - we use a diamond shape as the uvs are rotated
  const mask = sdDiamond(_uv).toVar()
  mask.assign(step(mask, maskSize))

  return mask.mulAssign(slash)
})

/**
 * Draws a square with a circular hole in the center, using box and sphere SDFs.
 * @param {vec2} _uv - The UV coordinates.
 * @param {number} [size=0.4] - Size of the outer square.
 * @param {number} [circleScalar=0.75] - Scalar for the inner circle size (relative to square).
 * @returns {float} Pattern mask value (0-1).
 */
const squareCirclePattern = Fn(([_uv, size = 0.4, circleScalar = 0.75]) => {
  const circleSize = float(size).mul(circleScalar)

  const outerSquare = sdBox2d(_uv).toVar()
  outerSquare.assign(step(outerSquare, size))

  const innerCircle = sdSphere(_uv).toVar()
  innerCircle.assign(smoothstep(circleSize, circleSize.add(0.005), innerCircle).oneMinus())

  return outerSquare.subAssign(innerCircle)
})

/**
 * Creates a ring around a small circle, using ring and sphere SDFs.
 * @param {vec2} _uv - The UV coordinates.
 * @param {number} [size=0.4] - Radius of the outer ring.
 * @param {number} [ringThickness=0.025] - Thickness of the ring.
 * @param {number} [circleSize=0.1] - Radius of the inner circle.
 * @returns {float} Pattern mask value (0-1).
 */
const circleRingPattern = Fn(([_uv, size = 0.4, ringThickness = 0.025, circleSize = 0.1]) => {
  const outerRing = sdRing(_uv, size).toVar()
  outerRing.assign(smoothstep(ringThickness, ringThickness + 0.005, outerRing).oneMinus())

  const innerCircle = sdSphere(_uv).toVar()
  innerCircle.assign(smoothstep(circleSize, circleSize + 0.005, innerCircle).oneMinus())

  return innerCircle.addAssign(outerRing)
})

/**
 * Draws two concentric rings using ring SDFs at different radii.
 * @param {vec2} _uv - The UV coordinates.
 * @param {number} [ringThickness=0.025] - Thickness of both rings.
 * @returns {float} Pattern mask value (0-1).
 */
const doubleRingPattern = Fn(([_uv, ringThickness = 0.025]) => {
  const outerRing = sdRing(_uv, 0.4).toVar()
  outerRing.assign(smoothstep(ringThickness, ringThickness + 0.005, outerRing).oneMinus())

  const innerRing = sdRing(_uv, 0.15).toVar()
  innerRing.assign(smoothstep(ringThickness, ringThickness + 0.005, innerRing).oneMinus())

  return innerRing.addAssign(outerRing)
})

/**
 * Combines a hexagon and a circle, both masked and blended using SDFs.
 * @param {vec2} _uv - The UV coordinates.
 * @param {number} [size=0.25] - Size of the hexagon.
 * @param {number} [circleSize=0.25] - Radius of the circle.
 * @returns {float} Pattern mask value (0-1).
 */
const hexagonCirclePattern = Fn(([_uv, size = 0.25, circleSize = 0.25]) => {
  const h = sdHexagon(_uv, size).toVar()
  h.assign(smoothstep(0.1, 0.105, h))

  const c = sdSphere(_uv).toVar()
  c.assign(smoothstep(circleSize, circleSize + 0.005, c).oneMinus())

  return h.addAssign(c)
})

/**
 * Draws a corner pattern by combining a square and two lines, then inverting and adding the square.
 * @param {vec2} _uv - The UV coordinates.
 * @param {number} [size=0.25] - Size of the inner square.
 * @param {number} [thickness=0.1] - Thickness of the lines.
 * @returns {float} Pattern mask value (0-1).
 */
const cornerPattern = Fn(([_uv, size = 0.25, thickness = 0.1]) => {
  const innerSquare = sdBox2d(_uv).toVar()
  innerSquare.assign(step(innerSquare, size))

  const verticalLine = sdLine(_uv.x).toVar()
  verticalLine.assign(step(verticalLine, thickness))
  const horizontalLine = sdLine(_uv.y).toVar()
  horizontalLine.assign(step(horizontalLine, thickness))

  const invertedCross = clamp(0, 1, verticalLine.add(horizontalLine)).toVar()
  invertedCross.addAssign(innerSquare)

  return invertedCross
})

/**
 * Creates a cross pattern in the corners and center using rotated and non-rotated UVs, combining lines and clamping to form the pattern.
 * @param {vec2} _uv - The UV coordinates (centered, usually in [-1, 1]).
 * @param {number} [thickness=0.25] - Thickness of the cross lines.
 * @param {number} [width=0.98] - Width of the rotated cross.
 * @returns {float} Pattern mask value (0-1).
 */
const crossSquareCornerPattern = Fn(([_uv, thickness = 0.25, width = 0.98]) => {
  const rotatedUv = rotate(_uv, 0.25 * Math.PI)

  const verticalLine = sdLine(_uv.x).toVar()
  verticalLine.assign(step(verticalLine, thickness))
  const horizontalLine = sdLine(_uv.y).toVar()
  horizontalLine.assign(step(horizontalLine, thickness))

  const invertedCross = clamp(0, 1, verticalLine.add(horizontalLine)).oneMinus()

  const crossVerticalLine = sdLine(rotatedUv.x).oneMinus().toVar()
  crossVerticalLine.assign(step(width, crossVerticalLine))
  const crossHorizontalLine = sdLine(rotatedUv.y).oneMinus().toVar()
  crossHorizontalLine.assign(step(width, crossHorizontalLine))

  const cross = clamp(0, 1, crossVerticalLine.add(crossHorizontalLine))

  const pattern = clamp(0, 1, invertedCross.add(cross))

  return pattern
})

/**
 * Draws a square corner pattern by combining vertical and horizontal lines, then inverting the cross.
 * @param {vec2} _uv - The UV coordinates.
 * @param {number} [thickness=0.25] - Thickness of the lines.
 * @returns {float} Pattern mask value (0-1).
 */
const cornerSquarePattern = Fn(([_uv, thickness = 0.25]) => {
  const verticalLine = sdLine(_uv.x).toVar()
  verticalLine.assign(step(verticalLine, thickness))
  const horizontalLine = sdLine(_uv.y).toVar()
  horizontalLine.assign(step(horizontalLine, thickness))

  const invertedCross = clamp(0, 1, verticalLine.add(horizontalLine)).oneMinus()

  return invertedCross
})

/**
 * Generates a diagonal hatch pattern by rotating UVs and creating repeated smooth lines.
 * @param {vec2} _uv - The UV coordinates.
 * @param {number} [repeat=5] - Number of hatch lines per unit.
 * @param {number} [thickness=0.4] - Thickness of each hatch line.
 * @returns {float} Pattern mask value (0-1).
 */
const hatchPattern = Fn(([_uv, repeat = 5, thickness = 0.4]) => {
  // Rotate uvs by 45 degrees
  const rotatedUv = rotate(_uv, 0.25 * Math.PI)

  // Create a repeating pattern of lines - these will appear rotated due to our domain rotation above
  const pattern = fract(rotatedUv.x.mul(repeat)).toVar()
  pattern.assign(smoothstep(thickness, float(thickness).add(0.005), pattern))

  return pattern
})

/**
 * Creates a cross with a square in the center, using rotated UVs for the cross and a box SDF for the square.
 * @param {vec2} _uv - The UV coordinates.
 * @param {number} [crossLength=0.3] - Length of the cross arms.
 * @param {number} [squareSize=0.23] - Size of the central square.
 * @returns {float} Pattern mask value (0-1).
 */
const squareCrossPattern = Fn(([_uv, crossLength = 0.3, squareSize = 0.23]) => {
  const rotatedUv = rotate(_uv, 0.25 * Math.PI)

  const crossThickness = float(crossLength).div(2)

  const crossLeft = max(abs(rotatedUv.x), abs(rotatedUv.y).sub(crossLength)).toVar()
  crossLeft.assign(step(crossLeft, crossThickness))

  const crossRight = max(abs(rotatedUv.x).sub(crossLength), abs(rotatedUv.y)).toVar()
  crossRight.assign(step(crossRight, crossThickness))

  const pattern = clamp(0, 1, crossLeft.addAssign(crossRight))

  const middleSquare = sdBox2d(_uv).toVar()
  middleSquare.assign(step(middleSquare, squareSize))

  pattern.subAssign(middleSquare)

  return pattern
})

/**
 * Draws a cross inside a circle, masking the cross with a square and the whole with a circular mask.
 * @param {vec2} _uv - The UV coordinates.
 * @param {number} [circleSize=0.4] - Radius of the outer circle mask.
 * @param {number} [squareSize=0.175] - Size of the central square mask.
 * @param {number} [width=0.95] - Width of the cross lines.
 * @returns {float} Pattern mask value (0-1).
 */
const crossCirclePattern = Fn(([_uv, circleSize = 0.4, squareSize = 0.175, width = 0.95]) => {
  const rotatedUv = rotate(_uv, 0.25 * Math.PI)

  const middleSquare = sdBox2d(_uv).toVar()
  middleSquare.assign(step(middleSquare, squareSize))

  const verticalLine = sdLine(rotatedUv.x).oneMinus().toVar()
  verticalLine.assign(step(width, verticalLine))
  const horizontalLine = sdLine(rotatedUv.y).oneMinus().toVar()
  horizontalLine.assign(step(width, horizontalLine))
  verticalLine.addAssign(horizontalLine)

  middleSquare.addAssign(verticalLine)

  const circleMask = sdSphere(_uv).toVar()
  circleMask.assign(smoothstep(circleSize, circleSize + 0.005, circleMask).oneMinus())

  return circleMask.subAssign(middleSquare)
})

/**
 * Combines a hexagon and an equilateral triangle, both masked and blended using SDFs.
 * @param {vec2} _uv - The UV coordinates.
 * @param {number} [size=0.25] - Size of the hexagon.
 * @param {number} [triangleSize=0.2] - Size of the triangle.
 * @returns {float} Pattern mask value (0-1).
 */
const hexagonTrianglePattern = Fn(([_uv, size = 0.25, triangleSize = 0.2]) => {
  const h = sdHexagon(_uv, size).toVar()
  h.assign(smoothstep(0.1, 0.105, h))

  const t = sdEquilateralTriangle(_uv, triangleSize).toVar()
  t.assign(smoothstep(0.1, 0.105, t).oneMinus())

  return h.addAssign(t)
})

const patterns = Fn(() => {
  // Get our aspect-correct UVs
  const _uv = screenAspectUV(screenSize)

  const finalColor = vec3(0).toVar()

  // Comment out the shape you want to use
  // finalColor.addAssign(plusPattern(_uv))
  // finalColor.addAssign(crossPattern(_uv))
  // finalColor.addAssign(slashPattern(_uv))
  // finalColor.addAssign(squareCirclePattern(_uv))
  // finalColor.addAssign(circleRingPattern(_uv))
  // finalColor.addAssign(doubleRingPattern(_uv))
  // finalColor.addAssign(hexagonCirclePattern(_uv))
  // finalColor.addAssign(cornerPattern(_uv))
  // finalColor.addAssign(crossSquareCornerPattern(_uv))
  // finalColor.addAssign(cornerSquarePattern(_uv))
  // finalColor.addAssign(hatchPattern(_uv))
  // finalColor.addAssign(squareCrossPattern(_uv))
  // finalColor.addAssign(crossCirclePattern(_uv))
  finalColor.addAssign(hexagonTrianglePattern(_uv))

  return finalColor
})

export default patterns
