import { Fn, screenSize, vec3, step, time, oscSine, float, vec2, remap } from 'three/tsl'
import { screenAspectUV } from '@/tsl/utils/function/screen_aspect_uv'
import { sdBox2d, sdSphere } from '@/tsl/utils/sdf/shapes'
import { smin } from '@/tsl/utils/sdf/operations'
import WebGPUSketch from '@/components/sketch/webgpu_sketch.js'

const sminSandbox = Fn(() => {
  // Get our aspect-correct UVs
  const _uv = screenAspectUV(screenSize)

  // Multiply the time by 0.1 to slow down the animation
  const _time = time.mul(0.1)

  // The oscSine() TSL function gives us an oscillating value between 0 and 1 using time. This is equivalent to sin(_time).mul(0.5).add(0.5)
  const s = oscSine(_time)

  // We're using the remap() function to remap the sine value to a range of 0 to 0.15. This is equivalent to s.mul(0.15), but it's good to demonstrate
  const smoothing = remap(s, float(0.0), float(1.0), float(0.0), float(0.15))

  // Create a base color to return - note the .toVar() function here. This is a TSL function that converts the value to a variable so we can assign it later
  const finalColor = vec3(0.0).toVar()

  // Create our shapes - 3 circles, and a square
  const sdC1 = sdSphere(vec2(_uv.x.add(0.1), _uv.y.sub(0.2)), 0.15).toVar()
  const sdC2 = sdSphere(vec2(_uv.x.sub(0.2), _uv.y.sub(0.1)), 0.08).toVar()
  const sdC3 = sdSphere(vec2(_uv.x.sub(0.23), _uv.y.add(0.22)), 0.12).toVar()
  const sdB1 = sdBox2d(vec2(_uv.x.add(0.15), _uv.y.add(0.26)), 0.03).toVar()

  // Combine the shapes using smin() and our animated smoothing factor
  const p = smin(sdC1, sdC2, smoothing).toVar()

  // Combine the result from the first operation, with the third circle
  p.assign(smin(p, sdC3, smoothing))

  // Combine the result from the first operation, with the square
  p.assign(smin(p, sdB1, smoothing))

  // Step the result to create a sharp edge
  p.assign(step(0.1, p))

  finalColor.assign(p)

  return finalColor
})

const canvas = document.querySelector('#webgpu-canvas')
const sketch = new WebGPUSketch(canvas, sminSandbox())

export default sketch
